Wald <- function(x,
    model,
    R=NULL,
    q=NULL){
  ans <- if(is.list(x) ||
     is.list(R) ||
     is.list(q)
    ) Wald.list(x,model,R,q)
  else Wald.formula(x,model,R,q)
  ans$call <- match.call()
  ans
}


Wald.formula <- function(
    x,
    model,
    R=NULL,
    q=NULL){
    formula <- x
    call <- match.call()
    call[[1]] <- as.symbol("Wald")
    test.terms <- terms(formula)
    model.terms <- terms(model)
    test.labels <- attr(test.terms,"term.labels")
    model.labels <- attr(model.terms,"term.labels")  
    if(!all(test.labels %in% model.labels)) stop("cannot test terms not in model")
    test.ix <- getCoefNames(model,test.labels)
    
    totest <- coef(model)[test.ix]
    VCov <- vcov(model)[test.ix,test.ix]
    if(!length(R)) R <- diag(length(test.ix))
    if(!is.matrix(R)) R <- t(as.matrix(R))
    if(!length(q)) q <- rep(0,nrow(R))
    RVR <- R%*%vcov(model)[test.ix,test.ix]%*%t(R)
    crit <- R%*%totest-q
    RVR.qr <- qr(RVR)
    df <- RVR.qr$rank
    if(df == ncol(RVR)) chisq <- sum(crit*solve(RVR.qr,crit))
    else chisq <- sum(crit*(MASS::ginv(RVR)%*%crit))
    pval <- pchisq(chisq,df,lower.tail=FALSE)
    return(
        structure(list(
        Chisq=chisq,
        df=df,
        pval=pval,
        tested=crit,
        terms=test.terms,
        model.call=model$call,
        call=call
        ),
        class="Wald"
    ))
}

Wald.list <- function(x,model,R=NULL,q=NULL){
    if(inherits(x,"formula")) x <- list(x)
    model <-list(model)
    call <- match.call()
    call[[1]] <- as.symbol("Wald")
    if(is.matrix(R)) R <- list(R)
    if(is.numeric(q)) q <- list(q)
    tests <- suppressWarnings(mapply(Wald.formula,x,model,R,q))
    chisq <- unlist(tests["Chisq",])
    df <- unlist(tests["df",])
    pval <- unlist(tests["pval",])
    names(chisq) <- colnames(tests)
    if(is.list(model) && length(names(model))==ncol(tests))
        names(chisq) <- names(model)
    if(is.list(R) && length(names(R))==ncol(tests))
        names(chisq) <- names(R)
    if(is.list(q) && length(names(q))==ncol(tests))
        names(chisq) <- names(q)
    return(
        structure(list(
        Chisq=chisq,
        df=df,
        pval=pval,
        tested=tests["tested",],
        terms=tests["terms",],
        model.call=model$call,
        call=call
        ),
        class="Wald"
    ))
}

print.Wald <- function(x,digits = max(3, getOption("digits") - 3),
        signif.stars = getOption("show.signif.stars"),...){
    cat("\nCall:\n")
    cat(paste(deparse(x$call), sep = "\n", collapse = "\n"),
        "\n\n", sep = "")
    testtab <- as.matrix(x)
    printCoefmat(testtab, digits=digits, tst.ind=1 ,cs.ind=NULL, signif.stars = signif.stars,
            P.values=TRUE,has.Pvalue=TRUE,
            na.print = "NA", ...)
    invisible(x)
}

as.matrix.Wald <- function(x,...){
    ans <- do.call(cbind,x[c("Chisq","df","pval")])
    if(length(x$Chisq)>1){
        if(length(names(x$Chisq))) 
            rownames(ans) <- names(x$Chisq)
        else
            rownames(ans) <- sapply(x$terms,function(t) paste(deparse(t,width=500)))
        }
    else rownames(ans) <- ""
    colnames(ans) <- c("W","df","Pr(>W)")
    ans
}

getCoefNames <- function(object,term.labels){
    res <- strsplit(term.labels,split=":")
    for(i in seq_along(res)){
        tl <- lapply(res[[i]],function(v)paste(v,object$xlevels[[v]],sep=""))
        if(length(tl)>1){
           for(j in 2:length(tl))
            tl[[1]] <- outer(tl[[1]],tl[[j]],paste,sep=":")
           res[[i]] <- c(tl[[1]])
        }
    }
    res <- unlist(res)
    nc <- names(coef(object))
    res <- nc %in% res
    nc[res]
}