MLMaster <- function(condition, reps, Ncores, loc, seed, icc, cli, clusts, probit, cdiff, DFApprox) {
	
	## This is the master function that does the simulation and can be called repeadedly
	# Definitions:
	# condition - Which condition is it?
	# reps - Number of replications
	# Ncores - if large computation cluster is available: how many cores shall be used?
	# cli - With cross-level interaction?
	# clusts - How many upper level units?
	# loc - Where is the home directory?
	# icc - Which degree of intraclass correlation?

	# Let's start: Define the seed (from: XYZ)
	set.seed(seed)

	# Choose appropriate directory for simulation data
	if (probit) {
		dataloc <- str_c(loc, 'RSimulations/Probit/')
	} else if (cdiff == 0) {
		dataloc <- str_c(loc, 'RSimulations/Linear/CDiff-0/')
	} else {
		dataloc <- str_c(loc, 'RSimulations/Linear/CDiff-05/')
	}


	# Save every (N of cores) * 10 replications,
	# so we run simulations by running a loop with
	# ceil(reps/(Ncores/10)) replications
	rounds <- floor(reps/(Ncores * 10)) 
	# --> We use floor rather than ceil b/c we start looping at zero
	print("Let's go!")

	for (round in 0:rounds) {
		
		# This loop ensures that we save every 
		# Ncores * 10 replications

		# How many replications to add in this round?
		# If round < rounds, we want Ncores * 10
		# but in the final round, fewer may remain

		if (round < rounds) {
			loop <- Ncores * 10
		} else {
			loop <- reps - (Ncores * 10 * rounds)
		}

		# Progress report 
		print(Ncores * 10 * round)

		# --> Only run the following lines if loop > 0  
		# If reps is a multiple of (Ncores * 10), no replications remain to be run in the final round

		if (loop > 0) {

			# Generate loop times multilevel data.frames
			# This is where random element is introduced and it is not parallelized
			# So no need to worry about using different seeds for different worker processes
			ml.data <- replicate(loop, 
				MLData(icc = icc, cli = cli, clusts = clusts,
					probit = probit, cdiff = cdiff),
				simplify = "array")
			ml.data <- lapply(ml.data, as.data.frame)

			# Set up for parallel estimation
			# psocketing under Windows and forking otherwise


			if (str_detect(Sys.info()["sysname"],"Windows")) {
				cl <- makeCluster(rep("localhost", Ncores), type = "PSOCK")
				# Load libraries on worker processes (not needed with forking)
				clusterEvalQ(cl, library(dplyr)) 
				clusterEvalQ(cl, library(lme4)) 
				clusterEvalQ(cl, library(lmerTest)) 
				clusterEvalQ(cl, library(stringr)) 
				clusterEvalQ(cl, library(hglm)) 
			} else {
				mcaffinity(1:Ncores)
				cl <- makeForkCluster(Ncores)
			}

			# Use parLapply to apply multi.modeller function to list of (simulated) data frames
			sims <- parLapply(cl, ml.data, multi.modeller, cli = cli, loc = loc, probit = probit, DFApprox = DFApprox)
			stopCluster(cl)

			# Array to long dataframe
			simulations <- sims[[1]]
			for (i in 2:loop) {
				simulations <- rbind(simulations, sims[[i]])
			}
			simulations <- as.data.frame(simulations)

			if (probit) {
				if (cli) {
					colnames(simulations) <- c('beta.cons', 'se.cons', 't.cons', 'pval.zbased.cons',
					'beta.x', 'se.x', 't.x', 'pval.zbased.x',
					'beta.z', 'se.z', 't.z', 'pval.zbased.z',
					'beta.x.z', 'se.x.z', 't.x.z', 'pval.zbased.x.z',
					'convinfo1', 'convinfo2',
					'sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x',
					'estimator')
				} else {
					colnames(simulations) <- c('beta.cons', 'se.cons', 't.cons', 'pval.zbased.cons',
					'beta.x', 'se.x', 't.x', 'pval.zbased.x',
					'beta.z', 'se.z', 't.z', 'pval.zbased.z',
					'convinfo1', 'convinfo2',
					'sd.cons', 'sd.resid',
					'estimator')
				}
			} else {
				if (DFApprox) {
					if (cli) {
						colnames(simulations) <- c('beta.cons', 'se.cons', 'dfsat.cons', 't.cons',
						'beta.x', 'se.x', 'dfsat.x', 't.x',
						'beta.z', 'se.z', 'dfsat.z', 't.z',
						'beta.x.z', 'se.x.z', 'dfsat.x.z', 't.x.z',
						'convinfo1', 'convinfo2',
						'sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x',
						'estimator')
					} else {
						colnames(simulations) <- c('beta.cons', 'se.cons', 'dfsat.cons', 't.cons',
						'beta.x', 'se.x', 'dfsat.x', 't.x',
						'beta.z', 'se.z', 'dfsat.z', 't.z',
						'convinfo1', 'convinfo2',
						'sd.cons', 'sd.resid',
						'estimator')
					}
				} else {
					if (cli) {
						colnames(simulations) <- c('beta.cons', 'se.cons',
						'beta.x', 'se.x',
						'beta.z', 'se.z',
						'beta.x.z', 'se.x.z',
						'convinfo1', 'convinfo2',
						'sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x',
						'estimator')
					} else {
						colnames(simulations) <- c('beta.cons', 'se.cons',
						'beta.x', 'se.x',
						'beta.z', 'se.z',
						'convinfo1', 'convinfo2',
						'sd.cons', 'sd.resid',
						'estimator')
					}
				}
			}

			# Attach info about experimental condition
			simulations$icc    <- icc
			simulations$cli    <- cli
			simulations$clusts <- clusts

			# Replication counter
			simulations$repit <- rep(((round * Ncores * 10) + (1:loop)), each = 2)

			# Filename for output file
			if (cli) {
				fname <- str_c(dataloc,'Cond',condition,'_icc',icc,'_CLI','_clusts',clusts,'_cdiff',cdiff,'_DFApprox',DFApprox,'.rds')
			} else {
				fname <- str_c(dataloc,'Cond',condition,'_icc',icc,'_NoCLI','_clusts',clusts,'_cdiff',cdiff,'_DFApprox',DFApprox,'.rds')
			}


			# Append previous replications for the same experimental condition and save
			if (round == 0) {
				saveRDS(simulations, fname)
			} else {
				simulations <- rbind(simulations, readRDS(fname))
				saveRDS(simulations, fname)
			}

			# Under Windows, we encountered memory issues when running function repeatedly
			# (i.e. for different conditions)
			# Try to resolve by explicitly removing simulations and issuing garbage collect

			rm(simulations, sims, ml.data)
			gc()

			# Short sleep
			Sys.sleep(1)

		}
	}
	
	# Progress report: Done (with one experimental condition)
	print("Done!")
}
