MLData = function(icc, cli, clusts, probit, cdiff) {
	
	# Function that generates multilevel data
	# Mimics the set-up of Stegmueller 2013
	# clust: is the number of upper-level clusters
	# ICC: Share of variance that is between clusters, intraclass correlation
	# cli: includes a cross-level interaction?
	# Returns: a dataframe with multi-level data
	library("mvtnorm")

	## 1. Generate the upper-level data
	df           <- as.data.frame(matrix(1, nrow = clusts, ncol = 7))
	colnames(df) <- c('cl.id', 'y', 'z', 'x', 'rs.x', 'u', 'e')

	# 1.1 The cluster ID
	df$cl.id <- 1:clusts

	# Set var of u so that icc is 5, 10, or 15%
	# Set cov(rs.x, u) so that correlation is 0.3872984,
	# which is the value in Stegmueller 2013 for CLI-setting (his "effects type V/VI")
	# where he only considers an ICC of 10
	# var of random slope is .3 for linear and probit
	# .3872984 = .1 / (sqrt(.2222222) * sqrt(.3))
	# values for other conditions found by
	# .3872984 * sqrt(var(u)) * sqrt(var(.3)) = cov_rsx_u

	if (probit) {
		# For probit model, Stegmueller uses var(e) = 1 (consistent with model)
		# Pick var(u) accordingly 
		if (icc == 5) {
			var_u     <- .05263158
			cov_rsx_u <- .04866643
		}
		if (icc == 10) {
			var_u     <- .1111111
			cov_rsx_u <- .07071069
		}
			if (icc == 15) {
			var_u     <- .1764706
			cov_rsx_u <- .08911329
		}
	} else {
		# For linear model, Stegmueller uses var(e) = 2
		# Pick var(u) accordingly
		if (icc == 5) {
			var_u     <- .1052632
			cov_rsx_u <- .06882475
		}
		if (icc == 10) {
			var_u     <- .2222222
			cov_rsx_u <- .1
		}
			if (icc == 15) {
			var_u     <- .3529412
			cov_rsx_u <- .1260252
		}

	}

	# 1.2 The upper-level predictor z and the
	# random intercept (variance = 0.222222 for an ICC of ~ 10%)
	# + random slope (variance = 0.3)
	# covariance between random intercept and slope = .1
	df$z <- rnorm(n = clusts, mean = 0, sd = 1)
	vcovmat <- matrix(c(.3, cov_rsx_u, cov_rsx_u, var_u), nrow = 2, ncol = 2)
	df[, c("rs.x", "u")] <- rmvnorm(clusts, mean = c(0, 0), sigma = vcovmat)

	# Introduce compositional differences if cdiff > 0
	# (cdiff = % of variance in x that is between clusters)
	# by shifting cluster means away from 0
	shift_sd <- sqrt(cdiff/(1-cdiff))
	df$shift_x <- rnorm(clusts, mean = 0, sd = shift_sd)
	# Also works for cdiffs = 0 because sqrt(0/(1-0)) = 0 
	# & rnorm(clusts, 0, 0) returns clusts times 0
	# cdiff/(1-cdiff) = x <==> cdiff = x/(1-x) 
	# (Proportion of variance that is between clusters, within cluster variance is 0 see below)

	## 2. Generate the lower-level data
	
	# Expand upper-level data by 500
	df <- df[rep(seq_len(dim(df)[1]), 500), ]

	# 2.2 Generate lower-level predictor x & lower-level error
	# x is N(0,1)
	df$x <- rnorm(n = clusts*500, mean = 0, sd = 1) + df$shift_x

	# When we add compositional differences (i.e., cdiff > 0)
	# Total (= between + within) variance of x becomes greater than 1
	# We rescale x, so its total variance is 1 (as in Stegmueller's setup)
	df$x = df$x * sqrt(1/(1 + shift_sd^2))
	
	# Residual error has var of 2 for linear and var of 1 for probit
	if (probit) {
		df$e <- rnorm(n = clusts*500, mean = 0, sd = 1)
	} else {
		df$e <- rnorm(n = clusts*500, mean = 0, sd = sqrt(2))		
	}

	# 2.3 Generate dependent variable
	# Two variants (with and without CLI)
	if (cli) {
		df$y <- with(df,
			(.25 + .1 * z + rs.x) * x + .2 * z + u + e)
	} else {
		df$y <- with(df,
			.25 * x + .2 * z + u + e)
	}
	# Probit version needs to be dichotomized
	if (probit) df$y <- ifelse(df$y < 0, 0, 1)

	## 3. Finish and spit out data
	df <- list(as.matrix(df))
	df
}
