multi.modeller = function (ml.data, cli, loc, probit, DFApprox) {
	## Function that estimates the models of interst for our multilevel data
	## from sim_02_multidata.r
	# Arguments:
	# ml.data is the simulated multi-level dataset
	# The other arguments are the estimation formulas, i.e. the models

	# Load our functions to extract estimates of interest
	source(str_c(loc,"RJobs/sim_03a_estGetters.R"))

	## Let's get started:
	# The data
	mul.data <- as.data.frame(ml.data)

	## 1. Set-up
	# 1.1 The estimation formulas
	if (cli) {
		formula <- 'y ~ x*z + (1 + x | cl.id)'
	} else {
		formula <- 'y ~ x + z + (1 | cl.id)'
	}

	# 1.3 The output dataset: estimates
	# What is extracted depends on model (no Satterthwaite approx. for probit)
	if (probit) {
		if (cli) {
			estimates             <- matrix(NA, nrow = 2, ncol = 23)
			colnames(estimates)   <- c('beta.cons', 'se.cons', 't.cons', 'pval.zbased.cons',
				'beta.x', 'se.x', 't.x', 'pval.zbased.x',
				'beta.z', 'se.z', 't.z', 'pval.zbased.z',
				'beta.x.z', 'se.x.z', 't.x.z', 'pval.zbased.x.z',
				'convinfo1', 'convinfo2',
				'sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x',
				'estimator')
		} else {
			estimates             <- matrix(NA, nrow = 2, ncol = 17)
			colnames(estimates)   <- c('beta.cons', 'se.cons', 't.cons', 'pval.zbased.cons',
				'beta.x', 'se.x', 't.x', 'pval.zbased.x',
				'beta.z', 'se.z', 't.z', 'pval.zbased.z',
				'convinfo1', 'convinfo2',
				'sd.cons', 'sd.resid',
				'estimator')
		}
		estimates[, "estimator"] <- c("REML/EQL (hglm::hglm2)", "ML (lme4::glmer)")
	} else {
		if (DFApprox) {
			if (cli) {
     			estimates             <- matrix(NA, nrow = 2, ncol = 23)
      			colnames(estimates)   <- c('beta.cons', 'se.cons', 'dfsat.cons', 't.cons',
			        'beta.x', 'se.x', 'dfsat.x', 't.x',
			        'beta.z', 'se.z', 'dfsat.z', 't.z',
			        'beta.x.z', 'se.x.z', 'dfsat.x.z', 't.x.z',
			        'convinfo1', 'convinfo2',
			        'sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x',
			        'estimator')
     		} else {
        		estimates             <- matrix(NA, nrow = 2, ncol = 17)
        		colnames(estimates)   <- c('beta.cons', 'se.cons', 'dfsat.cons', 't.cons',
			        'beta.x', 'se.x', 'dfsat.x', 't.x',
			        'beta.z', 'se.z', 'dfsat.z', 't.z',
			        'convinfo1', 'convinfo2',
			        'sd.cons', 'sd.resid',
			        'estimator')
      		}
      		estimates[, "estimator"] <- c("REML", "ML")
    	} else {
			if (cli) {
				estimates             <- matrix(NA, nrow = 2, ncol = 15)
				colnames(estimates)   <- c('beta.cons', 'se.cons',
					'beta.x', 'se.x',
					'beta.z', 'se.z',
					'beta.x.z', 'se.x.z',
					'convinfo1', 'convinfo2',
					'sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x',
					'estimator')
			} else {
				estimates             <- matrix(NA, nrow = 2, ncol = 11)
				colnames(estimates)   <- c('beta.cons', 'se.cons',
					'beta.x', 'se.x',
					'beta.z', 'se.z',
					'convinfo1', 'convinfo2',
					'sd.cons', 'sd.resid',
					'estimator')
			}
			estimates[, "estimator"] <- c("REML", "ML")
		}
	}

	## 2 Estimation
	
	# Probit
	if (probit) {
		
		# Fit hglm model (REML), this is the same with and without CLI
		# (formual was customized above)
		# In a few cases there are problems other than non-convergence/warnings
		# What I've seen is that var(e) = 0, which creates problmes 
		# because log of var(e) is modelled
		# Use tryCatch to prevent termination of loop in these cases
		hglm.fit <- tryCatch(
			{
			hglm::hglm2(meanmodel = eval(parse(text = formula)), data = mul.data, family = binomial(link = "probit"))
			},
			error = function(cond) {
				# message(cond) # Message stops execution of parallel computations
				return(NULL)
			}
		)

		# Extract and store results and estimate ML model (glmer)
		# Details differ between case with and case without CLI 
		if (cli) { # CLI case
		
			# Extract REML results
			if (is.null(hglm.fit)) {
				estimates[1, c('convinfo1', 'convinfo2')] <- "Severe error; no results"
			} else {
				# Write results of interest to estimates object
				estimates[1, c('beta.cons', 'beta.x', 'beta.z', 'beta.x.z')] <- hglm.fit$fixef
				if (hglm.fit$Converge == "converged") estimates[1, c('se.cons', 'se.x', 'se.z', 'se.x.z')] <- hglm.fit$SeFe # SEs N/A if model did not converge
				if (hglm.fit$Converge == "converged") estimates[1, c('t.cons', 't.x', 't.z', 't.x.z')]         <- hglm.fit$fixef/hglm.fit$SeFe
				if (hglm.fit$Converge == "converged") estimates[1, c('pval.zbased.cons', 'pval.zbased.x', 'pval.zbased.z', 'pval.zbased.x.z')] <- 2*pnorm(abs(hglm.fit$fixef/hglm.fit$SeFe), lower.tail = FALSE)
				estimates[1, c('sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x')] <- c(sqrt(hglm.fit$varRanef), NA, NA)
				estimates[1, c('convinfo1', 'convinfo2')] <- c(hglm.fit$Converge, NA)
			}

			# Estimate ML model using glmer
			# Stegmueller (2013) uses 15 integration points, 
			# but glmer only permits nAGQ > 1 with single random effect,
			# so not an option for CLI model
			glmer.fit <- lme4::glmer(formula = formula, data = mul.data, family = binomial(link = "probit"))
			fixefs.glmer  <- coef(summary(glmer.fit))

			# Write results to estimates object
			estimates[2, c('beta.cons', 'beta.x', 'beta.z', 'beta.x.z')] <- fixefs.glmer[,1]
			estimates[2, c('se.cons', 'se.x', 'se.z', 'se.x.z')]         <- fixefs.glmer[,2]
			estimates[2, c('t.cons', 't.x', 't.z', 't.x.z')]         <- fixefs.glmer[,3]
			estimates[2, c('pval.zbased.cons', 'pval.zbased.x', 'pval.zbased.z', 'pval.zbased.x.z')]         <- fixefs.glmer[,4]
			estimates[2, c('sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x')] <- vcGetter(glmer.fit, cli = cli)
			estimates[2, c('convinfo1', 'convinfo2')] <- convGetter(glmer.fit)

		} else {

			# DCE case (no CLI)
			if (is.null(hglm.fit)) {
				estimates[1, c('convinfo1', 'convinfo2')] <- "Severe error; no results"
			} else {
				# Write results of interest to estimates object
				estimates[1, c('beta.cons', 'beta.x', 'beta.z')] <- hglm.fit$fixef
				if (hglm.fit$Converge == "converged") estimates[1, c('se.cons', 'se.x', 'se.z')]         <- hglm.fit$SeFe # SEs N/A if model did not converge
				if (hglm.fit$Converge == "converged") estimates[1, c('t.cons', 't.x', 't.z')]         <- hglm.fit$fixef/hglm.fit$SeFe
				if (hglm.fit$Converge == "converged") estimates[1, c('pval.zbased.cons', 'pval.zbased.x', 'pval.zbased.z')] <- 2*pnorm(abs(hglm.fit$fixef/hglm.fit$SeFe), lower.tail = FALSE)
				estimates[1, c('sd.cons', 'sd.resid')] <- c(sqrt(hglm.fit$varRanef), NA)
				estimates[1, c('convinfo1', 'convinfo2')] <- c(hglm.fit$Converge, NA)
			}

			# Estimate ML model, this is random intercept, so we can use 15 integration points
			glmer.fit <- lme4::glmer(formula = formula, data = mul.data,
				family = binomial(link = "probit"), nAGQ = 15)
			fixefs.glmer  <- coef(summary(glmer.fit))

			# Write to estimates
			estimates[2, c('beta.cons', 'beta.x', 'beta.z')] <- fixefs.glmer[,1]
			estimates[2, c('se.cons', 'se.x', 'se.z')]         <- fixefs.glmer[,2]
			estimates[2, c('t.cons', 't.x', 't.z')]         <- fixefs.glmer[,3]
			estimates[2, c('pval.zbased.cons', 'pval.zbased.x', 'pval.zbased.z')]         <- fixefs.glmer[,4]
			estimates[2, c('sd.cons', 'sd.resid')] <- vcGetter(glmer.fit, cli = cli)
			estimates[2, c('convinfo1', 'convinfo2')] <- convGetter(glmer.fit)
		}

		# Clean up 
		rm(hglm.fit, glmer.fit, fixefs.glmer)
		gc()

	} else { # This is the linear case now

		# Do we want DF approximations (we do not do it for all
		# conditions b/c it's time consuming)

		if (DFApprox) {
			
			# REML fit using lmerTest (which implements Satterthwaite approx)
			lmer.fit.reml <- do.call(lmerTest::lmer, list(formula = formula, data = mul.data, REML = TRUE))
			fixefs.reml   <- coef(summary(lmer.fit.reml))
			convinfo.reml <- convGetter(lmer.fit.reml)
			# The variance components
			vcs.reml      <- vcGetter(lmer.fit.reml, cli = cli)

			# ML
			lmer.fit.ml <- lme4::lmer(formula = formula, data = mul.data, REML = FALSE)
			fixefs.ml   <- coef(summary(lmer.fit.ml))
			convinfo.ml <- convGetter(lmer.fit.ml)
			# The variance components
			vcs.ml      <- vcGetter(lmer.fit.ml, cli = cli)

			# Write to estimates according, details depend on presence of CLI
	
			if (cli) {
				estimates[1, c('beta.cons', 'beta.x', 'beta.z', 'beta.x.z')] <- fixefs.reml[,1]
				estimates[1, c('se.cons', 'se.x', 'se.z', 'se.x.z')]         <- fixefs.reml[,2]
				estimates[2, c('beta.cons', 'beta.x', 'beta.z', 'beta.x.z')] <- fixefs.ml[,1]
				estimates[2, c('se.cons', 'se.x', 'se.z', 'se.x.z')]         <- fixefs.ml[,2]
				
				# In a few  cases lmerTest returns standard lmer results w/o
				# df and p-values (because Satterthwaite approx. did not work)
				# fixefs then only has three columns (b, se, t-value), need special code for this situation
				if (ncol(fixefs.reml) > 3) {
					estimates[1, c('dfsat.cons', 'dfsat.x', 'dfsat.z', 'dfsat.x.z')] <- fixefs.reml[,3]
					estimates[1, c('t.cons', 't.x', 't.z', 't.x.z')]                 <- fixefs.reml[,4]
				} else {
					estimates[1, c('t.cons', 't.x', 't.z', 't.x.z')] <- fixefs.reml[,3]
				}

				# Extract variance component estimates
				estimates[1, c('sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x')] <- vcGetter(lmer.fit.reml, cli = cli)
				estimates[2, c('sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x')] <- vcGetter(lmer.fit.ml, cli = cli)
			
			} else { # DCE case
				estimates[1, c('beta.cons', 'beta.x', 'beta.z')] <- fixefs.reml[,1]
				estimates[1, c('se.cons', 'se.x', 'se.z')]       <- fixefs.reml[,2]
				estimates[2, c('beta.cons', 'beta.x', 'beta.z')] <- fixefs.ml[,1]
				estimates[2, c('se.cons', 'se.x', 'se.z')]       <- fixefs.ml[,2]
	
				if (ncol(fixefs.reml) > 3) {
					estimates[1, c('dfsat.cons', 'dfsat.x', 'dfsat.z')] <- fixefs.reml[,3]
					estimates[1, c('t.cons', 't.x', 't.z')] <- fixefs.reml[,4]
				} else {
					estimates[1, c('t.cons', 't.x', 't.z')] <- fixefs.reml[,3]
				}
	
				estimates[1, c('sd.cons', 'sd.resid')] <- vcGetter(lmer.fit.reml, cli = cli)
				estimates[2, c('sd.cons', 'sd.resid')] <- vcGetter(lmer.fit.ml, cli = cli)
			}

			# Clean up
			rm(lmer.fit.reml, lmer.fit.ml, fixefs.reml, vcs.reml,
				fixefs.ml, vcs.ml,  convinfo.reml, convinfo.ml)
			gc()
		
		} else { # Without DF approximation

			# REML
			lmer.fit.reml <- lme4::lmer(formula = formula, data = mul.data, REML = TRUE)
			fixefs.reml   <- coef(summary(lmer.fit.reml))
			convinfo.reml <- convGetter(lmer.fit.reml)
			# The variance components
			vcs.reml      <- vcGetter(lmer.fit.reml, cli = cli)

			# ML
			lmer.fit.ml <- lme4::lmer(formula = formula, data = mul.data, REML = FALSE)
			fixefs.ml   <- coef(summary(lmer.fit.ml))
			convinfo.ml <- convGetter(lmer.fit.ml)
			# The variance components
			vcs.ml      <- vcGetter(lmer.fit.ml, cli = cli)

			# Write to results dataset, according to type of simulation (cli)
			if (cli) {
				estimates[1, c('beta.cons', 'beta.x', 'beta.z', 'beta.x.z')] <- fixefs.reml[,1]
				estimates[1, c('se.cons', 'se.x', 'se.z', 'se.x.z')]         <- fixefs.reml[,2]
				estimates[2, c('beta.cons', 'beta.x', 'beta.z', 'beta.x.z')] <- fixefs.ml[,1]
				estimates[2, c('se.cons', 'se.x', 'se.z', 'se.x.z')]         <- fixefs.ml[,2]
				estimates[1, c('sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x')] <- vcGetter(lmer.fit.reml, cli = cli)
				estimates[2, c('sd.cons', 'sd.x', 'sd.resid', 'corr.cons.x')] <- vcGetter(lmer.fit.ml, cli = cli)
			} else {
				estimates[1, c('beta.cons', 'beta.x', 'beta.z')] <- fixefs.reml[,1]
				estimates[1, c('se.cons', 'se.x', 'se.z')]       <- fixefs.reml[,2]
				estimates[2, c('beta.cons', 'beta.x', 'beta.z')] <- fixefs.ml[,1]
				estimates[2, c('se.cons', 'se.x', 'se.z')]       <- fixefs.ml[,2]
				estimates[1, c('sd.cons', 'sd.resid')] <- vcGetter(lmer.fit.reml, cli = cli)
				estimates[2, c('sd.cons', 'sd.resid')] <- vcGetter(lmer.fit.ml, cli = cli)
			}
			estimates[1, c('convinfo1', 'convinfo2')] <- convinfo.reml
			estimates[2, c('convinfo1', 'convinfo2')] <- convinfo.ml

			# Clean up
			rm(lmer.fit.reml, fixefs.reml, convinfo.reml, vcs.reml,
				lmer.fit.ml, fixefs.ml, convinfo.ml, vcs.ml)
			gc()
		}
	}
	# Spit out the estimates
	estimates
}

